function [dx] = PBR4BP(t, x, data)
%
% Planar Bicircular Restricted 4-Body Problem (PBR4BP)
%
%DESCRIPTION:
%This code provides state vector of the Planar Bicircular Restricted 4-Body
%in the Synodic r.f.
%
%PROTOTYPE
%   [dx] = PBR4BP(t, x, data)
%
%--------------------------------------------------------------------------
% INPUTS:
%   t          [1x1]       Time Instant              [-]
%   x          [4x1]       State Vector              [-]
%   data       [---]       System Data               [struct]
%--------------------------------------------------------------------------
% OUTPUTS:
%   dx         [4x1]       Derivative of State Vec.  [-]
%--------------------------------------------------------------------------
%
%NOTES:
% - The input "y" and output "dy" should be in the form:
%   x  = [x; y; vx; vy];
%   dx = [vx; vy; ax; ay];
% - The input "data" shall contain the following fields:
%    * data.mu  [1x1]  System Mass Parameter                [-]
%    * data.ms  [1x1]  3rd Body Mass                        [-] (usually Sun)
%    * data.rho [1x1]  3rd-(1st-2nd) Bodies Scaled Distance [-] (Sun-(Earth-Moon))
%    * data.oms [1x1]  Scaled Angular Velocity of 3rd Body  [-]
%
%CALLED FUNCTIONS:
% (none)
%
%UPDATES:
% (none)
%
%REFERENCES:
% [1] "On Optimal Two-Impulse Earth-Moon Transfers in a Four-Body Model",
%     Francesco Topputo.
%
%AUTHOR(s):
%Luigi De Maria, 2022
%

%% Equations of Motion

%Data
mu  = data.mu;
ms  = data.m3_adim; 
rho = data.DU_13_adim; 
oms = data.omega_3/data.omega_2 - 1; % velocità angolare Deimos nel sinodico
ths = oms*t + (data.th0D - data.th0P);

%Distances
r1 = sqrt((x(1)+mu)^2   + x(2)^2);
r2 = sqrt((x(1)+mu-1)^2 + x(2)^2);
r3 = sqrt((x(1) - rho*cos(ths))^2 + (x(2) - rho*sin(ths))^2);

%Compute Derivatives
%--- wrt x ---
a  = x(1) - (1-mu)*(x(1)+mu)/r1^3 - mu*(x(1)+mu-1)/r2^3; %dOM3/dx
b  = -(x(1) - rho*cos(ths)) / r3^3; %d(1/r3)/dx
c  = cos(ths); %d(xcos - ysin)/dx
%--- wrt y ---
d  = x(2) - (1-mu)*x(2)/r1^3 - mu*x(2)/r2^3; %dOM3/dy
e  = -(x(2) - rho*sin(ths)) / r3^3; %d(1/r3)/dy
f  = sin(ths);  %d(xcos - ysin)/dy

%State Derivative
dx = [ x(3);                                           % vx
       x(4);                                           % vy
       2*x(4) + a + ms*b - ms/rho^2*c;                 % ax
      -2*x(3) + d + ms*e - ms/rho^2*f];                % ay

end